<?php
/**
 * MultiChat WHMCS Integration - Hooks
 * 
 * Automated hooks for syncing clients with MultiChat
 */

use WHMCS\Database\Capsule;

if (!defined("WHMCS")) {
    die("This file cannot be accessed directly");
}

/**
 * Hook: ClientAdd
 * Triggered when a new client is created
 */
add_hook('ClientAdd', 1, function($vars) {
    $settings = getMultichatSettings();
    
    if ($settings['auto_link_customers'] != 'on') {
        return;
    }
    
    $clientId = $vars['userid'];
    
    try {
        require_once __DIR__ . '/multichat.php';
        multichat_sync_customer($clientId, $settings);
        
        logActivity('MultiChat: New client #' . $clientId . ' synced to MultiChat');
    } catch (Exception $e) {
        logActivity('MultiChat Error: Failed to sync client #' . $clientId . ' - ' . $e->getMessage());
    }
});

/**
 * Hook: ClientEdit
 * Triggered when client details are updated
 */
add_hook('ClientEdit', 1, function($vars) {
    $settings = getMultichatSettings();
    
    if ($settings['auto_link_customers'] != 'on') {
        return;
    }
    
    $clientId = $vars['userid'];
    
    try {
        require_once __DIR__ . '/multichat.php';
        multichat_sync_customer($clientId, $settings);
        
        logActivity('MultiChat: Client #' . $clientId . ' updated in MultiChat');
    } catch (Exception $e) {
        logActivity('MultiChat Error: Failed to update client #' . $clientId . ' - ' . $e->getMessage());
    }
});

/**
 * Hook: ClientLogin
 * Triggered when client logs into client area
 * Ensures they're synced before they use chat
 */
add_hook('ClientLogin', 1, function($vars) {
    $settings = getMultichatSettings();
    
    if ($settings['auto_link_customers'] != 'on') {
        return;
    }
    
    $clientId = $vars['userid'];
    
    // Check if already synced
    $linked = Capsule::table('mod_multichat_clients')
        ->where('client_id', $clientId)
        ->first();
    
    if (!$linked) {
        try {
            require_once __DIR__ . '/multichat.php';
            multichat_sync_customer($clientId, $settings);
        } catch (Exception $e) {
            logActivity('MultiChat Error: Failed to sync client #' . $clientId . ' on login - ' . $e->getMessage());
        }
    }
});

/**
 * Hook: ClientAreaFooterOutput
 * Inject chat widget in client area if enabled
 * Shows for both logged in and logged out users
 */
add_hook('ClientAreaFooterOutput', 1, function($vars) {
    $settings = getMultichatSettings();
    
    if ($settings['enable_client_portal'] != 'on' || empty($settings['bubble_key'])) {
        return '';
    }
    
    $multichatUrl = rtrim($settings['multichat_url'], '/');
    $bubbleKey = $settings['bubble_key'];
    
    // Try to get client data if logged in
    $client = Menu::context('client');
    
    if ($client) {
        // Logged in user - include auto-identify data
        $clientId = $client->id;
        $clientEmail = htmlspecialchars($client->email, ENT_QUOTES, 'UTF-8');
        $clientName = htmlspecialchars($client->fullName, ENT_QUOTES, 'UTF-8');
        
        return <<<HTML
<script>
(function() {
    var script = document.createElement('script');
    script.src = '{$multichatUrl}/app/widget.js';
    script.setAttribute('data-bubble-key', '{$bubbleKey}');
    script.setAttribute('data-whmcs-client-id', '{$clientId}');
    script.setAttribute('data-prefill-email', '{$clientEmail}');
    script.setAttribute('data-prefill-name', '{$clientName}');
    script.setAttribute('data-auto-identify', 'true');
    script.setAttribute('data-skip-form', 'true');
    document.head.appendChild(script);
})();
</script>
HTML;
    } else {
        // Guest user - basic widget only
        return <<<HTML
<script>
(function() {
    var script = document.createElement('script');
    script.src = '{$multichatUrl}/app/widget.js';
    script.setAttribute('data-bubble-key', '{$bubbleKey}');
    document.head.appendChild(script);
})();
</script>
HTML;
    }
});

/**
 * Hook: AdminAreaFooterOutput
 * Add quick access to MultiChat in admin area
 */
add_hook('AdminAreaFooterOutput', 1, function($vars) {
    $settings = getMultichatSettings();
    
    if ($settings['show_in_dashboard'] != 'on' || empty($settings['multichat_url'])) {
        return '';
    }
    
    $multichatUrl = rtrim($settings['multichat_url'], '/') . '/members/app-dashboard.php';
    
    return <<<HTML
<style>
.multichat-quick-link {
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: linear-gradient(135deg, #00cc33, #00ff44);
    color: white;
    padding: 12px 20px;
    border-radius: 50px;
    text-decoration: none;
    box-shadow: 0 4px 15px rgba(0,204,51,0.4);
    font-weight: 600;
    z-index: 9999;
    transition: all 0.3s ease;
}
.multichat-quick-link:hover {
    transform: translateY(-3px);
    box-shadow: 0 6px 20px rgba(0,204,51,0.5);
    color: white;
    text-decoration: none;
}
.multichat-quick-link i {
    margin-right: 8px;
}
</style>
<a href="{$multichatUrl}" target="_blank" class="multichat-quick-link">
    <i class="fas fa-comments"></i> MultiChat Dashboard
</a>
HTML;
});

/**
 * Helper: Get MultiChat Settings
 */
function getMultichatSettings() {
    $settings = [];
    $results = Capsule::table('tbladdonmodules')
        ->where('module', 'multichat')
        ->get();
    
    foreach ($results as $result) {
        $settings[$result->setting] = $result->value;
    }
    
    return $settings;
}
